import {
  BOOLEAN,
  CHAR,
  DATETIME,
  DECIMAL,
  EnumValue,
  INTEGER,
  LONG,
  TEXT,
  VALUE,
} from './base-value';
import { EntityObject } from './entity';
import {
  ActionMeta,
  AttrMeta,
  EntityMeta,
  LogicFlowMeta,
  TemplateMeta,
  contextDataMeta,
} from './meta';

// 组件上下文API，处理和上下文数据相关的逻辑，包括 数据源加载，字段获取、更新，事件逻辑流执行，数据向下传递修改

export enum LOAD_STATUS {
  UNLOAD,
  LOADING,
  LOADED,
}

export interface Filter {
  field: AttrMeta;
  type:
    | 'ISNULL'
    | 'NOTNULL'
    | 'EQ'
    | 'NEQ'
    | 'GT'
    | 'GTE'
    | 'LT'
    | 'LTE'
    | 'CONTAINS'
    | 'NOTCONTAIN'
    | 'STARTSWITH'
    | 'ENDSWITH'
    | 'BETWEEN';
  values: Array<BOOLEAN | TEXT | CHAR | INTEGER | LONG | DECIMAL | DATETIME | EnumValue>;
}

// 多个字段条件的 与 或 组件
export interface FilterGroup {
  type: 'AND' | 'OR';
  filters: Filter[];
}

// 多个组合的组合，用于支持 与 中有 或， 或者 或 中有 与
export interface FilterParam {
  type: 'AND' | 'OR';
  groups: FilterGroup[];
}

export interface SortParam {
  field: AttrMeta;
  type: 'DESC' | 'ASC' | 'DEFAULT';
}

export interface RetrieveParams {
  filter?: FilterParam;
  sort?: SortParam[];
  page?: {
    // 请求页面，从1开始
    number: number;
    // 每页大小
    size: number;
  };
}

export interface DataSource {
  // 处理数据源加载
  processDataSource: (options?: {
    onLoadingStart?: () => void;
    onLoadingFinish?: () => void;
  }) => void;

  // 过滤 分页
  configDataFilter: (params: RetrieveParams) => void;
  // 重新刷新
  refreshData: (options?: {
    reload?: boolean;
    onLoadingStart?: () => void;
    onLoadingFinish?: () => void;
  }) => void;
  // 设置加载的数据是否只读
  setReadOnly: (readOnly: boolean) => void;

  // 获取数据的总个数
  getTotalCount: () => number;
  // 获取当前数据分页的页码，主要场景是自定义逻辑流分页，有可能根据业务需要实际返回的页码并不是请求的页码
  getCurrentPageNumber: () => number;

  // 自定义逻辑流分页
  enableCustomPagination: (options: { pageSize: number }) => void;
  // 自定义逻辑流搜索
  enableCustomSearch: () => void;

  registerListener: (options: { loadListener?: () => void }) => void;
}

// 封装上下文的细节，使用户不用关心，这个字段是关联的字段还是 哪个数据容器的字段 甚至是个静态数据
export interface AttrValue<T extends VALUE> {
  get: () => T | undefined;
  set: (value: T) => void;
}

//组件上下文API
export interface PageContext {
  changeChildContextData: (
    child: any,
    data?: EntityObject | EntityObject[],
  ) => React.ReactElement | React.ReactElement[];

  getData: <T extends EntityObject | Array<EntityObject>>() => T | undefined;
  // 返回上层组件的上下文数据，对于非数据容器来说，和 getData返回值一样
  getParentData: <T extends EntityObject | Array<EntityObject>>() => T | undefined;
  isReadonly: (attrMeta?: AttrMeta | VALUE | contextDataMeta) => boolean;
  getDataSource: () => DataSource;
  getAttrValue: <T extends VALUE>(
    meta: AttrMeta | TemplateMeta | VALUE,
    entityObject?: EntityObject,
  ) => AttrValue<T>;
  executeAction: <
    T extends void | VALUE | Array<VALUE> | object | object[] | EntityObject | EntityObject[]
  >(
    meta: ActionMeta,
    options?: {
      eventParams?: Array<VALUE | Array<VALUE> | object | object[] | EntityObject | EntityObject[]>;
      popupCallback?: Function;
      popupCallbackParam?: any;
      noBack?: boolean;
      ignoreReadonly?: boolean;
    },
  ) => Promise<T>;

  getDataById: (dataId: string) => EntityObject | undefined;
  getDataByDbId: (entityId: string, dbId: string) => EntityObject | undefined;
  getDataByContextData: (contextData: string | undefined) => EntityObject | undefined;

  createEntityObject: (metaOrId: EntityMeta | string) => EntityObject | undefined;

  executeLogicFlow: (
    meta: LogicFlowMeta,
    options?: {
      logicFlowParams?: Array<
        VALUE | Array<VALUE> | object | object[] | EntityObject | EntityObject[]
      >;
    },
  ) => Promise<void | VALUE | Array<VALUE> | object | object[] | EntityObject | EntityObject[]>;

  executeExpression: (
    expression: string | TemplateMeta,
    options?: { currentObject?: EntityObject },
  ) => VALUE | Array<VALUE> | undefined;

  getCurrentWebPageType: () => 'PC_WEB' | 'H5_WEB' | 'MINIPROGRAM';
}

export type PropsWithPageContext<P = {}> = P & {
  pageContext?: PageContext;
  className?: string;
  nodeId?: string;
};
