import { ComponentApi } from '../api-v2/api';

type ApiFactory = (version: string) => ComponentApi;
export const API_VERSION = '2.0.0';

const getWindowInfo = (win: any, dep = 0): string => {
  if (dep > 5) {
    return `too many. dep is ${dep}`;
  }
  if (win) {
    return `(name is ${win.name} mobiCtx exist ${win.mobiCtx ? 'true' : 'false'} parent is ${
      win === win.parent ? 'self' : getWindowInfo(win.parent, dep + 1)
    })`;
  } else {
    return 'not exist';
  }
};

const getFactory = () => {
  let win = window;
  if (!(win as any).__mobi_api_factory && win.name === 'SimulatorRenderer') {
    win = win.parent as any;
  }
  if (!(win as any).__mobi_api_factory) {
    throw new Error(`Only support in Mobi application, window is ${getWindowInfo(window)}`);
  }
  if (typeof (win as any).__mobi_api_factory !== 'function') {
    throw new Error('win.__mobi_api_factory has been changed', (win as any).__mobi_api_factory);
  }
  return (win as any).__mobi_api_factory as ApiFactory;
};

let api: ComponentApi;
const getApi = () => {
  api = getFactory()(API_VERSION);

  if (!api) {
    throw new Error(
      `This component's version (${API_VERSION}) is higher than the platform. Upgrade the platform or use a lower version component api`,
    );
  }
  return api;
};

let apiCache: ComponentApi | undefined = undefined;

// 这里用proxy的目的是 延迟加载，如果直接导出api，会在模块加载时就调用工厂方法，而此时工厂方法可能还没有ready
const handler: ProxyHandler<ComponentApi> = {
  get(target, prop) {
    let tmp: any = undefined;
    try {
      tmp = getApi();
    } catch (err) {
      if (!tmp && apiCache) {
        // 在增加了多人协作后，应用被抢占时，页面引擎的iframe应该是被回收了
        // 当激活，再抢占回来时，页面引擎创建了新的iframe，而同时旧的iframe上的组件开始执行unmount，此时会调用一些api
        // 而旧的iframe应该是和parent失去了关联，导致无法获取到api实现，从而报错。
        // 所以这里使用缓存的api代替，避免页面报错。同时因为报错是可见的，所以怀疑抢占回来后 显示的还是旧iframe
        tmp = apiCache;
      } else {
        throw err;
      }
    }
    if (!apiCache && tmp) {
      apiCache = tmp;
    }
    return tmp[prop];
  },
};

const proxyApi = new Proxy({} as ComponentApi, handler);

export default proxyApi;
