import { FORMAT, FormatObj, PropRule } from './rule';

interface Props {
  formats?: (FORMAT | FormatObj)[]; // 该事件所使用的错误检查函数列表,默认为使用所有
  propName?: string; // 该事件所使用的prop,默认为'eventAction'
  alias?: string; // 该事件别名,默认为'点击触发动作'
  condition?: string; // 该事件错误检查开启条件,如'props.multiple'代表仅当'props.multiple'为true时执行这些错误检查
}

/**
 * 事件相关错误检查函数生成
 * @param {*} formats
 * @param {*} propName
 * @param {*} alias
 * @param {*} condition
 */
export const actionRules = ({
  formats = [
    FORMAT.logicCheck,
    FORMAT.logicParamsChangeCheck,
    FORMAT.workFlowEnvCheck,
    FORMAT.urlValidCheck,
    FORMAT.logicPageParamCheck,
    FORMAT.eventCustomPathRouterCheck,
    FORMAT.contextDataInsideJumpCheck,
    FORMAT.insideJumpCheck,
  ],
  propName = 'props.eventAction',
  alias = '点击触发动作',
  condition = undefined,
}: Props = {}) => {
  const rulesObj: Record<string, PropRule> = {
    logicCheck: {
      type: 'object',
      format: FORMAT.logicCheck,
      description: '绑定的逻辑流/工作流不存在，请重新绑定',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    logicParamsChangeCheck: {
      type: 'object',
      format: FORMAT.logicParamsChangeCheck,
      description: '绑定的逻辑流参数有变更，请重新绑定',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    workFlowEnvCheck: {
      type: 'object',
      format: FORMAT.workFlowEnvCheck,
      description: '工作流事件必须处于工作流上下文中',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    urlValidCheck: {
      type: 'object',
      format: FORMAT.urlValidCheck,
      description: '请输入正确的外部链接（以 http:// 或 https:// 为开头的正确地址）',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    logicPageParamCheck: {
      type: 'object',
      format: FORMAT.logicPageParamCheck,
      description: '绑定逻辑流的参数与实际页面嵌套参数不符',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    eventCustomPathRouterCheck: {
      type: 'object',
      format: FORMAT.eventCustomPathRouterCheck,
      description: '内部跳转不能绑定携带参数的自定义路由页面',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    contextDataInsideJumpCheck: {
      type: 'object',
      format: FORMAT.contextDataInsideJumpCheck,
      description: '选择的数据不存在/已更改，请重新选择',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    insideJumpCheck: {
      type: 'object',
      format: FORMAT.insideJumpCheck,
      description: '内部跳转页面不存在，请重新绑定',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
  };
  const targetRules: Array<PropRule> = [];
  formats.forEach((format) => {
    if (typeof format === 'string') {
      rulesObj[format] && targetRules.push(rulesObj[format]);
    } else {
      const { name, args, condition: formatCondition } = format;
      if (name && rulesObj[name]) {
        if (formatCondition) {
          targetRules.push({
            ...rulesObj[name],
            format: { name, args },
            condition: formatCondition,
          });
        } else {
          targetRules.push({ ...rulesObj[name], format: { name, args } });
        }
      }
    }
  });
  return targetRules;
};



