import { FORMAT, FormatObj, PropRule } from './rule';
interface Props {
  formats?: (FORMAT | FormatObj)[]; // 该字段所使用的错误检查函数列表,默认为使用所有
  propName?: string; // 该字段所使用的prop,默认为'value'
  alias?: string; // 该字段别名,默认为'配置字段'
  condition?: string; // 该字段错误检查开启条件,如'props.multiple'代表仅当'props.multiple'为true时执行这些错误检查
}

/**
 * 字段相关错误检查函数生成
 * @param {*} formats
 * @param {*} propName
 * @param {*} alias
 * @param {*} condition
 */
export const attrRules = ({
  formats = [
    FORMAT.contextDataCheck,
    FORMAT.fieldCheck,
    FORMAT.logicResultCheck,
    FORMAT.templateExpressionCheck,
  ],
  propName = 'props.value',
  alias = '配置字段',
  condition = undefined,
}: Props = {}) => {
  const rulesObj: Record<string, PropRule> = {
    contextDataCheck: {
      type: 'object',
      format: FORMAT.contextDataCheck,
      description: '选择的数据不存在/已更改，请重新选择',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    fieldCheck: {
      type: 'object',
      format: FORMAT.fieldCheck,
      description: '实体类型或组件层级已更改，该组件绑定的属性已错误',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    logicResultCheck: {
      type: 'object',
      format: FORMAT.logicResultCheck,
      description: '逻辑流返回值变化，该绑定字段已失效，请重新绑定父级数据容器的逻辑流',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
    templateExpressionCheck: {
      type: 'object',
      format: FORMAT.templateExpressionCheck,
      description: '模板表达式有误，请检查',
      propName,
      level: 'ERROR',
      alias,
      condition,
    },
  };
  const targetRules: Array<PropRule> = [];
  formats.forEach((format) => {
    if (typeof format === 'string') {
      rulesObj[format] && targetRules.push(rulesObj[format]);
    } else {
      const { name, args, condition: formatCondition } = format;
      if (name && rulesObj[name]) {
        if (formatCondition) {
          targetRules.push({
            ...rulesObj[name],
            format: { name, args },
            condition: formatCondition,
          });
        } else {
          targetRules.push({ ...rulesObj[name], format: { name, args } });
        }
      }
    }
  });
  return targetRules;
};
